<?php

declare(strict_types=1);


namespace Drupal\tailwind_jit;

use Drupal\Component\Utility\Crypt;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\File\FileSystemInterface;
use Psr\Log\LoggerInterface;
use Drupal\Core\Site\Settings;
use Symfony\Component\Process\Process;

/**
 * Wrapper class for the executable Tailwind CSS binary.
 */
class Compiler {

  /**
   * The key of the settings variable containing the path to the executable.
   *
   * @var string
   */
  protected const SETTINGS_KEY_EXECUTABLE = 'tailwind_jit_executable';

  /**
   * The key of the settings variable containing the timeout for the compiler.
   *
   * @var string
   */
  protected const SETTINGS_KEY_TIMEOUT = 'tailwind_jit_timeout';

  /**
   * The key of the settings variable controlling minification of CSS output.
   *
   * @var string
   */
  protected const SETTINGS_KEY_MINIFY = 'tailwind_jit_minify';

  /**
   * The path to the compiler executable.
   *
   * @var string
   */
  protected ?string $executable;

  /**
   * The timeout for the shell executable.
   *
   * @var int
   */
  protected int $timeout;

  /**
   * Minify the compiled CSS.
   *
   * @var bool
   */
  protected bool $minify;

  /**
   * The filesystem service.
   *
   * @var \Drupal\Core\File\FileSystemInterface
   */
  protected FileSystemInterface $fileSystem;

  /**
   * The logger service.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected LoggerInterface $logger;

  /**
   * The configuration factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected ConfigFactoryInterface $configFactory;

  /**
   * The constructor.
   */
  public function __construct(LoggerInterface $logger, FileSystemInterface $file_system, ConfigFactoryInterface $config_factory) {
    $this->logger = $logger;
    $this->fileSystem = $file_system;
    $this->configFactory = $config_factory;
    $this->timeout = Settings::get(self::SETTINGS_KEY_TIMEOUT, 10);
    $this->minify = Settings::get(self::SETTINGS_KEY_MINIFY, TRUE);
    $this->executable = Settings::get(self::SETTINGS_KEY_EXECUTABLE);
  }

  /**
   * Checks the availability of the compiler by invoking the help command.
   *
   * @param string|null $output
   *   Output of the command line executable.
   * @param string|null $errorMessage
   *   Error message from the command line executable.
   *
   * @return int
   *   Return code from OS shell.
   */
  public function checkCompiler(string &$output = NULL, string &$errorMessage = NULL): int {
    if (!$this->executable) {
      $output = '';
      $errorMessage = strtr(
        "Path to executable not set. Configure \$settings['%key'] in your settings.php",
        ['%key' => self::SETTINGS_KEY_EXECUTABLE]
      );
      return -1;
    }
    return $this->runCompiler('--help', $output, $errorMessage);
  }

  /**
   * Invokes the Tailwind CSS command line executable.
   *
   * @param string $arguments
   *   Arguments to be passed to the command line executable.
   * @param string|null $output
   *   Output of the command line executable.
   * @param string|null $errorMessage
   *   Error message from the command line executable.
   *
   * @return int
   *   Return code from OS shell.
   */
  public function runCompiler(string $arguments, string &$output = NULL, string &$errorMessage = NULL): int {
    $commandLine = $this->executable . ' ' . $arguments;
    $output = '';
    $errorMessage = '';

    $process = Process::fromShellCommandline($commandLine, DRUPAL_ROOT);
    $process->setTimeout($this->timeout);
    try {
      $process->run();
      $output = $process->getOutput();
      $errorMessage = $process->getErrorOutput();
      $errorCode = $process->getExitCode();
    }
    catch (\Exception $e) {
      $errorMessage = $e->getMessage();
      $errorCode = $process->getExitCode() ?: 1;
    }
    if ($errorCode) {
      if ($output) {
        $this->logger->info($output);
      }
      if ($errorMessage) {
        $this->logger->error($errorMessage);
      }
    }
    return $errorCode;
  }

  /**
   * Generates Tailwind compiled CSS.
   *
   * @param string $content
   *   The HTML content used to generate the CSS.
   * @param string $cssInputFilename
   *   The CSS input file used to generate the CSS.
   * @param string $tailwindConfigFilename
   *   The Tailwind config file used to generate the CSS.
   * @param int|null $returnCode
   *   Return code from OS shell.
   *
   * @return string
   *   The compiled CSS generated by Tailwind,
   *   surrounded by a <style data-tailwind-jit="output"> tag
   */
  public function getCompiledCss(string $content, string $cssInputFilename = NULL, string $tailwindConfigFilename = NULL, int &$returnCode = NULL): string {
    if (!$this->executable) {
      $this->logger->error("Path to executable not set. Configure \$settings['%key'] in your settings.php", ['%key' => self::SETTINGS_KEY_EXECUTABLE]);
      $returnCode = -1;
      return '';
    }
    if ($cssInputFilename && !$this->fileSystem->realpath($cssInputFilename)) {
      $this->logger->error("CSS input file %file not found", ['%file' => $cssInputFilename]);
      $returnCode = -1;
      return '';
    }
    if ($tailwindConfigFilename && !$this->fileSystem->realpath($tailwindConfigFilename)) {
      $this->logger->error("Tailwind config file %file not found", ['%file' => $tailwindConfigFilename]);
      $returnCode = -1;
      return '';
    }
    if (!trim($content)) {
      $this->logger->warning("HTML content is empty.");
    }
    $randomFilename = 'tailwind_jit_' . Crypt::randomBytesBase64(32);
    $contentFilename = $this->fileSystem->createFilename("{$randomFilename}.html", $this->fileSystem->getTempDirectory());
    $tmpContentFile = $this->fileSystem->saveData($content, $contentFilename, FileSystemInterface::EXISTS_ERROR);
    if (!$tmpContentFile) {
      $this->logger->error("Failed creating temporary HTML content file.");
      $returnCode = -1;
      return '';
    }
    $cssOutputFilename = $this->fileSystem->createFilename("{$randomFilename}.css", $this->fileSystem->getTempDirectory());
    $arguments = "--content {$contentFilename} --output {$cssOutputFilename}";
    if ($cssInputFilename) {
      $arguments .= " --input {$cssInputFilename}";
    }
    if ($tailwindConfigFilename) {
      $arguments .= " --config {$tailwindConfigFilename}";
    }
    if ($this->minify) {
      $arguments .= ' --minify';
    }
    $returnCode = $this->runCompiler($arguments);
    if ($returnCode != 0) {
      $compiledCss = '';
    }
    else {
      $compiledCss = '<style data-tailwind-jit="output">' . file_get_contents($cssOutputFilename) . '</style>';
    }
    $this->fileSystem->delete($contentFilename);
    $this->fileSystem->delete($cssOutputFilename);
    return $compiledCss;
  }

}
