<?php

declare(strict_types=1);


namespace Drupal\tailwind_jit;

use Drupal\Core\Render\HtmlResponse;
use Psr\Log\LoggerInterface;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpKernel\HttpKernelInterface;

/**
 * Interceps HTML responses to inject Tailwind generated CSS on demand.
 */
class HttpMiddleware implements HttpKernelInterface {

  /**
   * The wrapped HTTP kernel.
   */
  protected HttpKernelInterface $httpKernel;

  /**
   * The tailwind_jit logger channel.
   */
  protected LoggerInterface $logger;

  /**
   * The wrapper for the commandline executable.
   */
  protected Compiler $compiler;

  /**
   * The constructor.
   */
  public function __construct(HttpKernelInterface $http_kernel, LoggerInterface $logger, Compiler $compiler) {
    $this->httpKernel = $http_kernel;
    $this->logger = $logger;
    $this->compiler = $compiler;
  }

  /**
   * {@inheritDoc}
   */
  public function handle(Request $request, $type = self::MASTER_REQUEST, $catch = TRUE) {
    $response = $this->httpKernel->handle($request, $type, $catch);

    // Only handle HTML responses for now.
    // @todo What about AjaxResponse?
    if (!($response instanceof HtmlResponse)) {
      return $response;
    }

    $content = $response->getContent();
    $linkTagRegex = '/<link\s+([^>]*\s+)*data-tailwind-jit\s*=\s*["\']?input["\']?[^>]*>/i';
    if (!preg_match($linkTagRegex, $content, $linkTag)) {
      return $response;
    }
    preg_match('/\s+href\s*=\s*["\']?([^"\']+\.css)["\']?/i', $linkTag[0], $hrefAttribute);
    $cssInputUri = $hrefAttribute[1] ?? FALSE;
    if (!$cssInputUri) {
      $this->logger->warning('Could not determine CSS input file.');
      return $response;
    }
    $cssInputFile = realpath(DRUPAL_ROOT . $cssInputUri);
    if (!$cssInputFile) {
      $this->logger->warning('CSS input file "%filename" not found in local filesystem.', ['%filename' => $cssInputUri]);
      return $response;
    }
    if (!str_starts_with($cssInputFile, DRUPAL_ROOT)) {
      $this->logger->warning('Can not compile CSS input file "%filename" because it is outside of DRUPAL_ROOT.', ['%filename' => $cssInputUri]);
      return $response;
    }

    $errorCode = 0;
    $compiledCss = $this->compiler->getCompiledCss($content, $cssInputFile, $errorCode);
    if ($errorCode) {
      return $response;
    }

    $content = preg_replace($linkTagRegex, $compiledCss, $content);
    $response->setContent($content);
    return $response;
  }

}
